<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author   SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZend_Controller
 * @category   SZend
 */

abstract class SZendControllerRequestAbstract
{
    /**
     * Has the action been dispatched?
     * @var boolean
     */
    protected $dispatched = false;

    /**
     * Module
     * @var string
     */
    protected $module;

    /**
     * Module key for retrieving module from params
     * @var string
     */
    protected $moduleKey = 'module';

    /**
     * Controller
     * @var string
     */
    protected $controller;

    /**
     * Controller key for retrieving controller from params
     * @var string
     */
    protected $controllerKey = 'controller';

    /**
     * Action
     * @var string
     */
    protected $action;

    /**
     * Action key for retrieving action from params
     * @var string
     */
    protected $actionKey = 'action';

    /**
     * Request parameters
     * @var array
     */
    protected $params = array();

    /**
     * Retrieve the module name
     *
     * @return string
     */
    public function getModuleName()
    {
        if (null === $this->module) {
            $this->module = $this->getParam($this->getModuleKey());
        }

        return $this->module;
    }

    /**
     * Get an action parameter
     *
     * @param string $key
     * @param mixed $default Default value to use if key not found
     * @return mixed
     */
    public function getParam($key, $default = null)
    {
        $key = (string)$key;
        if (array_key_exists($key, $this->params)) {
            return $this->params[$key];
        }

        return $default;
    }

    /**
     * Retrieve the module key
     *
     * @return string
     */
    public function getModuleKey()
    {
        return $this->moduleKey;
    }

    /**
     * Set the module key
     *
     * @param string $key
     * @return SZendControllerRequestAbstract
     */
    public function setModuleKey($key)
    {
        $this->moduleKey = (string)$key;

        return $this;
    }

    /**
     * Set the module name to use
     *
     * @param string $value
     * @return SZendControllerRequestAbstract
     */
    public function setModuleName($value)
    {
        $this->module = $value;

        return $this;
    }

    /**
     * Retrieve the controller name
     *
     * @return string
     */
    public function getControllerName()
    {
        if (null === $this->controller) {
            $this->controller = $this->getParam($this->getControllerKey());
        }

        return $this->controller;
    }

    /**
     * Retrieve the controller key
     *
     * @return string
     */
    public function getControllerKey()
    {
        return $this->controllerKey;
    }

    /**
     * Set the controller key
     *
     * @param string $key
     * @return SZendControllerRequestAbstract
     */
    public function setControllerKey($key)
    {
        $this->controllerKey = (string)$key;

        return $this;
    }

    /**
     * Set the controller name to use
     *
     * @param string $value
     * @return SZendControllerRequestAbstract
     */
    public function setControllerName($value)
    {
        $this->controller = $value;

        return $this;
    }

    /**
     * Retrieve the action name
     *
     * @return string
     */
    public function getActionName()
    {
        if (null === $this->action) {
            $this->action = $this->getParam($this->getActionKey());
        }

        return $this->action;
    }

    /**
     * Retrieve the action key
     *
     * @return string
     */
    public function getActionKey()
    {
        return $this->actionKey;
    }

    /**
     * Set the action key
     *
     * @param string $key
     * @return SZendControllerRequestAbstract
     */
    public function setActionKey($key)
    {
        $this->actionKey = (string)$key;

        return $this;
    }

    /**
     * Set the action name
     *
     * @param string $value
     * @return SZendControllerRequestAbstract
     */
    public function setActionName($value)
    {
        $this->action = $value;

        return $this;
    }

    /**
     * Retrieve only user params (i.e, any param specific to the object and not the environment)
     *
     * @return array
     */
    public function getUserParams()
    {
        return $this->params;
    }

    /**
     * Retrieve a single user param (i.e, a param specific to the object and not the environment)
     *
     * @param string $key
     * @param string $default Default value to use if key not found
     * @return mixed
     */
    public function getUserParam($key, $default = null)
    {
        if (array_key_exists($key, $this->params)) {
            return $this->params[$key];
        }

        return $default;
    }

    /**
     * Set an action parameter
     *
     * A $value of null will unset the $key if it exists
     *
     * @param string $key
     * @param mixed $value
     * @return SZendControllerRequestAbstract
     */
    public function setParam($key, $value)
    {
        $key = (string)$key;

        if ((null === $value) && array_key_exists($key, $this->params)) {
            unset($this->params[$key]);
        } elseif (null !== $value) {
            $this->params[$key] = $value;
        }

        return $this;
    }

    /**
     * Get all action parameters
     *
     * @return array
     */
    public function getParams()
    {
        return $this->params;
    }

    /**
     * Set action parameters en masse; does not overwrite
     *
     * Null values will unset the associated key.
     *
     * @param array $array
     * @return SZendControllerRequestAbstract
     */
    public function setParams(array $array)
    {
        $this->params = $this->params + (array)$array;

        foreach ($this->params as $key => $value) {
            if (null === $value) {
                unset($this->params[$key]);
            }
        }

        return $this;
    }

    /**
     * Determine if the request has been dispatched
     *
     * @return boolean
     */
    public function isDispatched()
    {
        return $this->dispatched;
    }

    /**
     * Set flag indicating whether or not request has been dispatched
     *
     * @param boolean $flag
     * @return SZendControllerRequestAbstract
     */
    public function setDispatched($flag = true)
    {
        $this->dispatched = $flag ? true : false;

        return $this;
    }
}
