<?php
/**
 * Configuration
 *
 * @author      Shopimind <contact@shopimind.com>
 * @copyright   Copyright (c) 2013 - IDVIVE SARL (http://www.idvive.com)
 * @license     New BSD license (http://license.idvive.com)
 * @package     ShopimindClient
 * @version     $Id Configuration.php 2013-04-24$
 */

class ShopimindClientConfiguration
{

    /**
     * Permet de configurer l'identifiant API
     *
     * @var string
     */
    protected $apiIdentification;

    /**
     * Permet de configurer un mot de passe pour l'API
     *
     * @var string
     */
    protected $apiPassword;

    /**
     * URL de connexion à l'api
     *
     * @var string
     */
    protected $apiUrl;

    /**
     * URL du client
     *
     * @var string
     */
    protected $urlClient;

    /**
     * Lang par défaut
     *
     * @var string
     */
    protected $defaultLang;

    /**
     * La defaultCurrency par défaut
     *
     * @var string
     */
    protected $defaultCurrency;

    /**
     * URL de page de contact
     *
     * @var string
     */
    protected $contactPage;

    /**
     * Numéro de téléphone du service client
     *
     * @var string
     */
    protected $phoneNumber;

    /**
     * Timezone de la boutique
     *
     * @var string
     */
    protected $timezone;

    /**
     * Multi-boutiques activé ?
     *
     * @var boolean
     */
    protected $multishopEnabled;

    /**
     * Id de la boutique (multi-boutiques)
     *
     * @var string
     */
    protected $shopIdShop;

    /**
     * Force de la synchro produits
     *
     * @var string
     */
    protected $forceSyncProducts;

    /**
     * Version du prestashop
     *
     * @var string
     */
    protected $prestashopVersion;

    /**
     * Version du module
     *
     * @var string
     */
    protected $moduleVersion;

    /**
     * Constructeur de l'objet config
     *
     * @return void
     */
    public function __construct()
    {
        $this->retrieveUrlClient();
    }

    /**
     * Va tanter de trouver l'url du client
     *
     * @return string
     */
    public function retrieveUrlClient()
    {
        $base_dir = dirname(__FILE__);
        $doc_root = preg_replace("!{$_SERVER['SCRIPT_NAME']}$!", '', $_SERVER['SCRIPT_FILENAME']);
        $doc_root_explode = explode("/", $doc_root);
        $last_dir_doc_root = $doc_root_explode[count($doc_root_explode)-1];
        $pos_last_dir = strripos($base_dir, $last_dir_doc_root);
        $base_url = Tools::substr($base_dir, $pos_last_dir+Tools::strlen($last_dir_doc_root));
        $protocol = empty($_SERVER['HTTPS']) ? 'http' : 'https';
        $port = $_SERVER['SERVER_PORT'];
        $disp_port = ($protocol == 'http' && $port == 80 || $protocol == 'https' && $port == 443) ? '' : ":$port";
        $domain = $_SERVER['SERVER_NAME'];
        $url = "$protocol://{$domain}{$disp_port}{$base_url}";
        $url = preg_replace('#bin$#', '', $url);
        $this->setUrlClient($url);

        return $url;
    }

    /**
     * Retourne la tableau de configuration global
     *
     * @return array
     */
    public static function getGlobalConfig()
    {
        return array_merge_recursive(
            require dirname(__FILE__).'/../src/definitions.php',
            require dirname(__FILE__).'/../configuration.php'
        );
    }

    /**
     * Permet de construire l'objet Configuration
     *
     * @param string $identifiantAPI
     * @param string $passwordAPI
     * @param string $defaultLang
     * @param string $defaultCurrency
     * @return ShopimindClientConfiguration
     */
    public static function factory(
        $identifiantAPI,
        $passwordAPI,
        $defaultLang,
        $defaultCurrency,
        $contactPage = null,
        $phoneNumber = null,
        $timezone = null,
        $multishop_enabled = null,
        $shop_id_shop = null,
        $forceSyncProducts = false,
        $prestashopVersion = null,
        $moduleVersion = null
    ) {
        $config = new self;
        $config->setApiIdentification($identifiantAPI)
            ->setApiPassword($passwordAPI)
            ->setDefaultLang($defaultLang)
            ->setDefaultCurrency($defaultCurrency)
            ->setContactPage($contactPage)
            ->setPhoneNumber($phoneNumber)
            ->setTimezone($timezone)
            ->setMultishopEnabled($multishop_enabled)
            ->setShopIdShop($shop_id_shop)
            ->setForceSyncProducts($forceSyncProducts)
            ->setPrestashopVersion($prestashopVersion)
            ->setModuleVersion($moduleVersion);

        return $config;
    }

    /**
     * Permet de connecter le serveur au module
     *
     * @return boolean
     */
    public function connectServer()
    {
        $data = array();

        if ($this->getDefaultLang() !== null) {
            $data['defaultLang'] = $this->getDefaultLang();
        }

        if ($this->getDefaultCurrency() !== null) {
            $data['defaultCurrency'] = $this->getDefaultCurrency();
        }

        if ($this->getContactPage() !== null) {
            $data['contactPage'] = $this->getContactPage();
        }

        if ($this->getPhoneNumber() !== null) {
            $data['phoneNumber'] = $this->getPhoneNumber();
        }

        if ($this->getUrlClient() !== null) {
            $data['urlClient'] = $this->getUrlClient();
        }

        if ($this->getTimezone() !== null) {
            $data['timezone'] = $this->getTimezone();
        }

        if ($this->getMultishopEnabled() !== null) {
            $data['multishopEnabled'] = $this->getMultishopEnabled();
        }

        if ($this->getShopIdShop() !== null) {
            $data['shopIdShop'] = $this->getShopIdShop();
        }

        if ($this->getForceSyncProducts() !== null) {
            $data['forceSyncProducts'] = $this->getForceSyncProducts();
        }

        if ($this->getPrestashopVersion() !== null) {
            $data['ecommerceVersion'] = $this->getPrestashopVersion();
        }

        if ($this->getModuleVersion() !== null) {
            $data['moduleVersion'] = $this->getModuleVersion();
        }

        $now = new \DateTime();
        $data['lastUpdateTimezone'] = $now->format('Y-m-d H:i:s');

        $data['langs'] = [];
        require_once dirname(__FILE__).'/../callback.php';
        if (method_exists('ShopimindClientCallback', 'getLangs')) {
            $data['langs'] = ShopimindClientCallback::getLangs(($this->getShopIdShop() !== null) ? $this->getShopIdShop() : false);
        }

        require_once dirname(__FILE__).'/../src/Client.php';
        $client = new ShopimindClientClient;
        $client->setRestService('connection');
        $client->setParameterPost($data);
        $response = $client->sendRequest('POST');

        if (is_array($response) && array_key_exists('success', $response) && $response['success'] === true) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Permet d'obtenir la langue par défaut de la boutique
     *
     * @return string
     */
    public function getDefaultLang()
    {
        return $this->defaultLang;
    }

    /**
     * Permet de modifier la langue par défaut de la boutique
     *
     * @param string $defaultLang
     * @return ShopimindClientConfiguration
     */
    public function setDefaultLang($defaultLang)
    {
        $this->defaultLang = $defaultLang;

        return $this;
    }

    /**
     * Permet d'obtenir la defaultCurrency de la boutique
     *
     * @return string
     */
    public function getDefaultCurrency()
    {
        return $this->defaultCurrency;
    }

    /**
     * Permet de modifier la defaultCurrency de la boutique
     *
     * @param string $defaultCurrency
     * @return ShopimindClientConfiguration
     */
    public function setDefaultCurrency($defaultCurrency)
    {
        $this->defaultCurrency = $defaultCurrency;

        return $this;
    }

    /**
     * Url de la page contact
     *
     * @return string
     */
    public function getContactPage()
    {
        return $this->contactPage;
    }

    /**
     * Modifier la page de contact de page
     *
     * @param string $contactPage
     * @return ShopimindClientConfiguration
     */
    public function setContactPage($contactPage)
    {
        $this->contactPage = $contactPage;

        return $this;
    }

    /**
     * Obtenir le numéro de téléphone du service client
     *
     * @return string
     */
    public function getPhoneNumber()
    {
        return $this->phoneNumber;
    }

    /**
     * Modifier le numéro de téléphone du service client
     *
     * @param string $phoneNumber
     * @return ShopimindClientConfiguration
     */
    public function setPhoneNumber($phoneNumber)
    {
        $this->phoneNumber = $phoneNumber;

        return $this;
    }

    /**
     * Permet d'obtenir l'url du ShopimindClient
     *
     * @return string
     */
    public function getUrlClient()
    {
        return $this->urlClient;
    }

    /**
     * Permet de modifier l'url du ShopimindClient
     *
     * @param string $urlClient
     * @return ShopimindClientConfiguration
     */
    public function setUrlClient($urlClient)
    {
        $this->urlClient = $urlClient;

        return $this;
    }

    /**
     * Permet de récupérer le timezone de la boutique
     *
     * @return string
     */
    public function getTimezone()
    {
        return $this->timezone;
    }

    /**
     * Permet de modifier le timezone de la boutique
     *
     * @param string $timezone
     * @return ShopimindClientConfiguration
     */
    public function setTimezone($timezone)
    {
        $this->timezone = $timezone;

        return $this;
    }

    /**
     * Permet de récupérer le statut de l'option multi-boutiques
     *
     * @return boolean
     */
    public function getMultishopEnabled()
    {
        return (bool)$this->multishopEnabled;
    }

    /**
     * Permet de modifier le statut de l'option multi-boutiques
     *
     * @param boolean $timezone
     * @return ShopimindClientConfiguration
     */
    public function setMultishopEnabled($multishopEnabled)
    {
        $this->multishopEnabled = (bool)$multishopEnabled;

        return $this;
    }

    /**
     * Permet de récupérer l'id de la boutique (multi-boutiques)
     *
     * @return string
     */
    public function getShopIdShop()
    {
        return $this->shopIdShop;
    }

    /**
     * Permet de modifier l'id de la boutique (multi-boutiques)
     *
     * @param string $shopIdShop
     * @return ShopimindClientConfiguration
     */
    public function setShopIdShop($shopIdShop)
    {
        $this->shopIdShop = $shopIdShop;

        return $this;
    }

    /**
     * Permet de savoir si on doit re-synchroniser les produits
     *
     * @return string
     */
    public function getForceSyncProducts()
    {
        return $this->forceSyncProducts;
    }

    /**
     * Permet d'indiquer si on doit re-synchroniser les produits
     *
     * @param string $forceSyncProducts
     * @return ShopimindClientConfiguration
     */
    public function setForceSyncProducts($forceSyncProducts)
    {
        $this->forceSyncProducts = $forceSyncProducts;

        return $this;
    }

    /**
     * Version Prestashop
     *
     * @return string
     */
    public function getPrestashopVersion()
    {
        return $this->prestashopVersion;
    }

    /**
     * Version Prestashop
     *
     * @param string $prestashopVersion
     * @return ShopimindClientConfiguration
     */
    public function setPrestashopVersion($prestashopVersion)
    {
        $this->prestashopVersion = $prestashopVersion;

        return $this;
    }

    /**
     * Version du module
     *
     * @return string
     */
    public function getModuleVersion()
    {
        return $this->moduleVersion;
    }

    /**
     * Version du module
     *
     * @param string $moduleVersion
     * @return ShopimindClientConfiguration
     */
    public function setModuleVersion($moduleVersion)
    {
        $this->moduleVersion = $moduleVersion;

        return $this;
    }

    /**
     * Permet de tester la communication avec le serveur
     *
     * @return boolean
     */
    public function testConnection()
    {
        require_once dirname(__FILE__).'/../src/Client.php';
        $client = new ShopimindClientClient;

        return $client->sayHello();
    }

    /**
     * Permet de lancer la configuration du client
     *
     * @return boolean
     */
    public function saveConfig()
    {
        return true;
/*        global $SHOPIMIND_CLIENT_CONFIGURATION;
        $lastConfig = require dirname(__FILE__).'/../configuration.php';

        if ($this->getApiIdentification() !== null) {
            $lastConfig['api']['identifiant'] = $this->getApiIdentification();
        }

        if ($this->getApiPassword() !== null) {
            $lastConfig['api']['password'] = $this->getApiPassword();
        }

        if ($this->getApiUrl() !== null) {
            $lastConfig['api']['url'] = $this->getApiUrl();
        }

        $content = "<?php\n"
            ."return ".var_export($lastConfig, true).";\n";

        if (file_put_contents(dirname(__FILE__).'/../configuration.php', $content) !== false) {
            $SHOPIMIND_CLIENT_CONFIGURATION = array_merge_recursive(
                require dirname(__FILE__).'/../src/definitions.php',
                require dirname(__FILE__).'/../configuration.php'
            );

            return true;
        }

        return false;*/
    }

    /**
     * Permet d'obtenir l'identifiant de l'API
     *
     * @return string
     */
    public function getApiIdentification()
    {
        return $this->apiIdentification;
    }

    /**
     * Permet de modifier l'identifiant de l'API
     *
     * @param string $apiIdentification
     * @return ShopimindClientConfiguration
     */
    public function setApiIdentification($apiIdentification)
    {
        $this->apiIdentification = $apiIdentification;

        return $this;
    }

    /**
     * Permet d'obtenir le mot de passe de l'API
     *
     * @return string
     */
    public function getApiPassword()
    {
        return $this->apiPassword;
    }

    /**
     * Permet de modifier le mot de passe de l'API
     *
     * @param string $apiPassword
     * @return ShopimindClientConfiguration
     */
    public function setApiPassword($apiPassword)
    {
        $this->apiPassword = $apiPassword;

        return $this;
    }

    /**
     * Permet d'obtenir l'url de l'api
     *
     * @return string
     */
    public function getApiUrl()
    {
        return $this->apiUrl;
    }

    /**
     * Permet de modifier l'url de l'api
     *
     * @param string $apiUrl
     * @return ShopimindClientConfiguration
     */
    public function setApiUrl($apiUrl)
    {
        $this->apiUrl = $apiUrl;

        return $this;
    }
}
