<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author    SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZendUri
 * @version    $Id: Uri.php 8064 2008-02-16 10:58:39Z thomas $
 * @category   SZend
 */

/**
 * @see SZendLoader
 */
require_once dirname(__FILE__).'/Loader.php';

abstract class SZendUri
{
    /**
     * Scheme of this URI (http, ftp, etc.)
     * @var string
     */
    protected $scheme = '';

    /**
     * SZendUri and its subclasses cannot be instantiated directly.
     * Use SZendUri::factory() to return a new SZendUri object.
     */
    abstract protected function __construct($scheme, $schemeSpecific = '');

    /**
     * Convenience function, checks that a $uri string is well-formed
     * by validating it but not returning an object.  Returns TRUE if
     * $uri is a well-formed URI, or FALSE otherwise.
     *
     * @param string $uri
     * @return boolean
     */
    public static function check($uri)
    {
        try {
            $uri = self::factory($uri);
        } catch (Exception $e) {
            return false;
        }

        return $uri->valid();
    }

    /**
     * Create a new SZendUri object for a URI.  If building a new URI, then $uri should contain
     * only the scheme (http, ftp, etc).  Otherwise, supply $uri with the complete URI.
     *
     * @param string $uri
     * @return SZendUri
     * @throws SZendUriException
     */
    public static function factory($uri = 'http')
    {
        /**
         * Separate the scheme from the scheme-specific parts
         * @link http://www.faqs.org/rfcs/rfc2396.html
         */
        $uri = explode(':', $uri, 2);
        $scheme = Tools::strtolower($uri[0]);
        $schemeSpecific = array_key_exists(1, $uri) ? $uri[1] : '';

        if (!Tools::strlen($scheme)) {
            require_once 'Uri/Exception.php';
            throw new SZendUriException('An empty string was supplied for the scheme');
        }

        // Security check: $scheme is used to load a class file, so only alphanumerics are allowed.
        if (!ctype_alnum($scheme)) {
            require_once 'Uri/Exception.php';
            throw new SZendUriException('Illegal scheme supplied, only alphanumeric characters are permitted');
        }

        /**
         * Create a new SZendUri object for the $uri. If a subclass of SZendUri exists for the
         * scheme, return an instance of that class. Otherwise, a SZendUriException is thrown.
         */
        switch ($scheme) {
            case 'http':
            case 'https':
                $className = 'SZendUriHttp';
                break;
            case 'mailto':
                // @todo
            default:
                require_once 'Uri/Exception.php';
                throw new SZendUriException("Scheme \"$scheme\" is not supported");
        }
        SZendLoader::loadClass($className);

        return new $className($scheme, $schemeSpecific);
    }

    /**
     * Return a string representation of this URI.
     *
     * @return  string
     * @see     getUri()
     */
    public function __toString()
    {
        return $this->getUri();
    }

    /******************************************************************************
     * Abstract Methods
     *****************************************************************************/

    /**
     * Return a string representation of this URI.
     *
     * @return string
     */
    abstract public function getUri();

    /**
     * Get the URI's scheme
     *
     * @return string|false Scheme or false if no scheme is set.
     */
    public function getScheme()
    {
        if (!empty($this->scheme)) {
            return $this->scheme;
        } else {
            return false;
        }
    }

    /**
     * Returns TRUE if this URI is valid, or FALSE otherwise.
     *
     * @return boolean
     */
    abstract public function valid();
}
