<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author    SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZendUri
 * @version    $Id: Http.php 8064 2008-02-16 10:58:39Z thomas $
 * @category   SZend
 */

/**
 * @see SZendUri
 */
require_once dirname(__FILE__).'/../Uri.php';

/**
 * @see SZendValidateHostname
 */
require_once dirname(__FILE__).'/../Validate/Hostname.php';

class SZendUriHttp extends SZendUri
{
    /**
     * URI parts are divided among these instance variables
     */
    protected $username = '';
    protected $password = '';
    protected $host = '';
    protected $port = '';
    protected $path = '';
    protected $query = '';
    protected $fragment = '';

    /**
     * Regular expression grammar rules for validation; values added by constructor
     */
    protected $regex = array();

    /**
     * Constructor accepts a string $scheme (e.g., http, https) and a scheme-specific part of the URI
     * (e.g., example.com/path/to/resource?query=param#fragment)
     *
     * @param string $scheme
     * @param string $schemeSpecific
     * @return void
     * @throws SZendUriException
     */
    protected function __construct($scheme, $schemeSpecific = '')
    {
        // Set the scheme
        $this->scheme = $scheme;

        // Set up grammar rules for validation via regular expressions. These
        // are to be used with slash-delimited regular expression strings.
        $this->regex['alphanum'] = '[^\W_]';
        $this->regex['escaped'] = '(?:%[\da-fA-F]{2})';
        $this->regex['mark'] = '[-_.!~*\'()\[\]]';
        $this->regex['reserved'] = '[;\/?:@&=+$,]';
        $this->regex['unreserved'] = '(?:'.$this->regex['alphanum'].'|'.$this->regex['mark'].')';
        $this->regex['segment'] = '(?:(?:'.$this->regex['unreserved'].'|'.$this->regex['escaped']
            .'|[:@&=+$,;])*)';
        $this->regex['path'] = '(?:\/'.$this->regex['segment'].'?)+';
        $this->regex['uric'] = '(?:'.$this->regex['reserved'].'|'.$this->regex['unreserved'].'|'
            .$this->regex['escaped'].')';
        // If no scheme-specific part was supplied, the user intends to create
        // a new URI with this object.  No further parsing is required.
        if (Tools::strlen($schemeSpecific) == 0) {
            return;
        }

        // Parse the scheme-specific URI parts into the instance variables.
        $this->parseUri($schemeSpecific);

        // Validate the URI
        if (!$this->valid()) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Invalid URI supplied');
        }
    }

    /**
     * Parse the scheme-specific portion of the URI and place its parts into instance variables.
     *
     * @param string $schemeSpecific
     * @return void
     * @throws SZendUriException
     */
    protected function parseUri($schemeSpecific)
    {
        // High-level decomposition parser
        $pattern = '~^((//)([^/?#]*))([^?#]*)(\?([^#]*))?(#(.*))?$~';
        $status = @preg_match($pattern, $schemeSpecific, $matches);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: scheme-specific decomposition failed');
        }

        // Failed decomposition; no further processing needed
        if (!$status) {
            return;
        }

        // Save URI components that need no further decomposition
        $this->path = (array_key_exists(4, $matches) && $matches[4]) ? $matches[4] : '';
        $this->query = (array_key_exists(6, $matches) && $matches[6]) ? $matches[6] : '';
        $this->fragment = (array_key_exists(8, $matches) && $matches[8]) ? $matches[8] : '';

        // Additional decomposition to get username, password, host, and port
        $combo = (array_key_exists(3, $matches) && $matches[3]) ? $matches[3] : '';
        $pattern = '~^(([^:@]*)(:([^@]*))?@)?([^:]+)(:(.*))?$~';
        $status = @preg_match($pattern, $combo, $matches);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: authority decomposition failed');
        }

        // Failed decomposition; no further processing needed
        if (!$status) {
            return;
        }

        // Save remaining URI components
        $this->username = (array_key_exists(2, $matches) && $matches[2]) ? $matches[2] : '';
        $this->password = (array_key_exists(4, $matches) && $matches[4]) ? $matches[4] : '';
        $this->host = (array_key_exists(5, $matches) && $matches[5]) ? $matches[5] : '';
        $this->port = (array_key_exists(7, $matches) && $matches[7]) ? $matches[7] : '';
    }

    /**
     * Validate the current URI from the instance variables. Returns true if and only if all
     * parts pass validation.
     *
     * @return boolean
     */
    public function valid()
    {
        /**
         * Return true if and only if all parts of the URI have passed validation
         */
        return $this->validateUsername()
            && $this->validatePassword()
            && $this->validateHost()
            && $this->validatePort()
            && $this->validatePath()
            && $this->validateQuery()
            && $this->validateFragment();
    }

    /**
     * Returns true if and only if the username passes validation. If no username is passed,
     * then the username contained in the instance variable is used.
     *
     * @param string $username
     * @return boolean
     * @throws SZendUriException
     */
    public function validateUsername($username = null)
    {
        if ($username === null) {
            $username = $this->username;
        }

        // If the username is empty, then it is considered valid
        if (Tools::strlen($username) == 0) {
            return true;
        }
        /**
         * Check the username against the allowed values
         *
         * @link http://www.faqs.org/rfcs/rfc2396.html
         */
        $status = @preg_match('/^('.$this->regex['alphanum'].'|'.$this->regex['mark'].'|'
            .$this->regex['escaped'].'|[;:&=+$,])+$/', $username);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: username validation failed');
        }

        return $status == 1;
    }

    /**
     * Returns true if and only if the password passes validation. If no password is passed,
     * then the password contained in the instance variable is used.
     *
     * @param string $password
     * @return boolean
     * @throws SZendUriException
     */
    public function validatePassword($password = null)
    {
        if ($password === null) {
            $password = $this->password;
        }

        // If the password is empty, then it is considered valid
        if (Tools::strlen($password) == 0) {
            return true;
        }

        // If the password is nonempty, but there is no username, then it is considered invalid
        if (Tools::strlen($password) > 0 && Tools::strlen($this->username) == 0) {
            return false;
        }

        /**
         * Check the password against the allowed values
         *
         * @link http://www.faqs.org/rfcs/rfc2396.html
         */
        $status = @preg_match('/^('.$this->regex['alphanum'].'|'.$this->regex['mark'].'|'
            .$this->regex['escaped'].'|[;:&=+$,])+$/', $password);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: password validation failed.');
        }

        return $status == 1;
    }

    /**
     * Returns true if and only if the host string passes validation. If no host is passed,
     * then the host contained in the instance variable is used.
     *
     * @param string $host
     * @return boolean
     * @uses SZend_Filter
     */
    public function validateHost($host = null)
    {
        if ($host === null) {
            $host = $this->host;
        }

        /**
         * If the host is empty, then it is considered invalid
         */
        if (Tools::strlen($host) == 0) {
            return false;
        }

        /**
         * Check the host against the allowed values; delegated to SZend_Filter.
         */
        $validate = new SZendValidateHostname(SZendValidateHostname::ALLOW_ALL);

        return $validate->isValid($host);
    }

    /**
     * Returns true if and only if the TCP port string passes validation. If no port is passed,
     * then the port contained in the instance variable is used.
     *
     * @param string $port
     * @return boolean
     */
    public function validatePort($port = null)
    {
        if ($port === null) {
            $port = $this->port;
        }

        // If the port is empty, then it is considered valid
        if (!Tools::strlen($port)) {
            return true;
        }

        // Check the port against the allowed values
        return ctype_digit((string)$port) && 1 <= $port && $port <= 65535;
    }

    /**
     * Returns true if and only if the path string passes validation. If no path is passed,
     * then the path contained in the instance variable is used.
     *
     * @param string $path
     * @return boolean
     * @throws SZendUriException
     */
    public function validatePath($path = null)
    {
        if ($path === null) {
            $path = $this->path;
        }
        /**
         * If the path is empty, then it is considered valid
         */
        if (Tools::strlen($path) == 0) {
            return true;
        }
        /**
         * Determine whether the path is well-formed
         */
        $pattern = '/^'.$this->regex['path'].'$/';
        $status = @preg_match($pattern, $path);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: path validation failed');
        }

        return (boolean)$status;
    }

    /**
     * Returns true if and only if the query string passes validation. If no query is passed,
     * then the query string contained in the instance variable is used.
     *
     * @param string $query
     * @return boolean
     * @throws SZendUriException
     */
    public function validateQuery($query = null)
    {
        if ($query === null) {
            $query = $this->query;
        }

        // If query is empty, it is considered to be valid
        if (Tools::strlen($query) == 0) {
            return true;
        }

        /**
         * Determine whether the query is well-formed
         *
         * @link http://www.faqs.org/rfcs/rfc2396.html
         */
        $pattern = '/^'.$this->regex['uric'].'*$/';
        $status = @preg_match($pattern, $query);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: query validation failed');
        }

        return $status == 1;
    }

    /**
     * Returns true if and only if the fragment passes validation. If no fragment is passed,
     * then the fragment contained in the instance variable is used.
     *
     * @param string $fragment
     * @return boolean
     * @throws SZendUriException
     */
    public function validateFragment($fragment = null)
    {
        if ($fragment === null) {
            $fragment = $this->fragment;
        }

        // If fragment is empty, it is considered to be valid
        if (Tools::strlen($fragment) == 0) {
            return true;
        }

        /**
         * Determine whether the fragment is well-formed
         *
         * @link http://www.faqs.org/rfcs/rfc2396.html
         */
        $pattern = '/^'.$this->regex['uric'].'*$/';
        $status = @preg_match($pattern, $fragment);
        if ($status === false) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('Internal error: fragment validation failed');
        }

        return (boolean)$status;
    }

    /**
     * Returns a URI based on current values of the instance variables. If any
     * part of the URI does not pass validation, then an exception is thrown.
     *
     * @return string
     * @throws SZendUriException
     */
    public function getUri()
    {
        if (!$this->valid()) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException('One or more parts of the URI are invalid');
        }
        $password = Tools::strlen($this->password) ? ":$this->password" : '';
        $auth = Tools::strlen($this->username) ? "$this->username$password@" : '';
        $port = Tools::strlen($this->port) ? ":$this->port" : '';
        $query = Tools::strlen($this->query) ? "?$this->query" : '';
        $fragment = Tools::strlen($this->fragment) ? "#$this->fragment" : '';

        return "$this->scheme://$auth$this->host$port$this->path$query$fragment";
    }

    /**
     * Returns the username portion of the URL, or FALSE if none.
     *
     * @return string
     */
    public function getUsername()
    {
        return Tools::strlen($this->username) ? $this->username : false;
    }

    /**
     * Sets the username for the current URI, and returns the old username
     *
     * @param string $username
     * @return string
     * @throws SZendUriException
     */
    public function setUsername($username)
    {
        if (!$this->validateUsername($username)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Username \"$username\" is not a valid HTTP username");
        }
        $oldUsername = $this->username;
        $this->username = $username;

        return $oldUsername;
    }

    /**
     * Returns the password portion of the URL, or FALSE if none.
     *
     * @return string
     */
    public function getPassword()
    {
        return Tools::strlen($this->password) ? $this->password : false;
    }

    /**
     * Sets the password for the current URI, and returns the old password
     *
     * @param string $password
     * @return string
     * @throws SZendUriException
     */
    public function setPassword($password)
    {
        if (!$this->validatePassword($password)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Password \"$password\" is not a valid HTTP password.");
        }
        $oldPassword = $this->password;
        $this->password = $password;

        return $oldPassword;
    }

    /**
     * Returns the domain or host IP portion of the URL, or FALSE if none.
     *
     * @return string
     */
    public function getHost()
    {
        return Tools::strlen($this->host) ? $this->host : false;
    }

    /**
     * Sets the host for the current URI, and returns the old host
     *
     * @param string $host
     * @return string
     * @throws SZendUriException
     */
    public function setHost($host)
    {
        if (!$this->validateHost($host)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Host \"$host\" is not a valid HTTP host");
        }
        $oldHost = $this->host;
        $this->host = $host;

        return $oldHost;
    }

    /**
     * Returns the TCP port, or FALSE if none.
     *
     * @return string
     */
    public function getPort()
    {
        return Tools::strlen($this->port) ? $this->port : false;
    }

    /**
     * Sets the port for the current URI, and returns the old port
     *
     * @param string $port
     * @return string
     * @throws SZendUriException
     */
    public function setPort($port)
    {
        if (!$this->validatePort($port)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Port \"$port\" is not a valid HTTP port.");
        }
        $oldPort = $this->port;
        $this->port = $port;

        return $oldPort;
    }

    /**
     * Returns the path and filename portion of the URL, or FALSE if none.
     *
     * @return string
     */
    public function getPath()
    {
        return Tools::strlen($this->path) ? $this->path : '/';
    }

    /**
     * Sets the path for the current URI, and returns the old path
     *
     * @param string $path
     * @return string
     * @throws SZendUriException
     */
    public function setPath($path)
    {
        if (!$this->validatePath($path)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Path \"$path\" is not a valid HTTP path");
        }
        $oldPath = $this->path;
        $this->path = $path;

        return $oldPath;
    }

    /**
     * Returns the query portion of the URL (after ?), or FALSE if none.
     *
     * @return string
     */
    public function getQuery()
    {
        return Tools::strlen($this->query) ? $this->query : false;
    }

    /**
     * Set the query string for the current URI, and return the old query
     * string This method accepts both strings and arrays.
     *
     * @param string|array $query The query string or array
     * @return string              Old query string
     */
    public function setQuery($query)
    {
        $oldQuery = $this->query;

        // If query is empty, set an empty string
        if (empty($query)) {
            $this->query = '';

            return $oldQuery;
        }

        // If query is an array, make a string out of it
        if (is_array($query)) {
            $query = http_build_query($query, '', '&');
            // If it is a string, make sure it is valid. If not parse and encode it
        } else {
            $query = (string)$query;
            if (!$this->validateQuery($query)) {
                parse_str($query, $query_array);
                $query = http_build_query($query_array, '', '&');
            }
        }

        // Make sure the query is valid, and set it
        if (!$this->validateQuery($query)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("'$query' is not a valid query string");
        }

        $this->query = $query;

        return $oldQuery;
    }

    /**
     * Returns the fragment portion of the URL (after #), or FALSE if none.
     *
     * @return string|false
     */
    public function getFragment()
    {
        return Tools::strlen($this->fragment) ? $this->fragment : false;
    }

    /**
     * Sets the fragment for the current URI, and returns the old fragment
     *
     * @param string $fragment
     * @return string
     * @throws SZendUriException
     */
    public function setFragment($fragment)
    {
        if (!$this->validateFragment($fragment)) {
            require_once 'SZend/Uri/Exception.php';
            throw new SZendUriException("Fragment \"$fragment\" is not a valid HTTP fragment");
        }
        $oldFragment = $this->fragment;
        $this->fragment = $fragment;

        return $oldFragment;
    }
}
